/**
 * MediaEdit
 * MediaSave
 * PanelBodyMedia
 * MediaUploadMedia
 */

import { linkOff } from '@wordpress/icons';

const { __ } = wp.i18n;

const {
    MediaPlaceholder,
    MediaUpload,
    MediaUploadCheck,
     __experimentalLinkControl: LinkControl
} = wp.blockEditor;

const { 
    PanelBody,
    ToggleControl,
    SelectControl,
    TextareaControl,
    Button,
    RangeControl,
    Flex,
    FlexItem,
    BaseControl
} = wp.components;


const { 
   useEffect, useState, Fragment
} = wp.element;

const { useSelect } = wp.data;

import { onSelectMedia, verticalPositioning } from './functions';

export const MediaEdit = ({
    mediaType,
    mediaUrl,
    mediaAlt,
    mediaId,
    mediaSizeSlug,
    mediaOrientation,
    mediaAnchor,
    relatedPostsLinkOption,
    relatedPosts,
    mediaCustomPosition,
    aspectRatio,
    originalAspectRatio,
    controls,
    autoplay,
    loop,
    muted,
    playsinline,
    preload,
    poster,
    figureProps,
    setAttributes
}) => {
    
    const addVerticalPositioning = verticalPositioning(mediaOrientation, originalAspectRatio, aspectRatio);

    const figureContent = (
        <>        
        {mediaType && mediaUrl ? (
            mediaType === 'image' ? (
                <img 
                    src={mediaUrl} 
                    alt={mediaAlt} 
                    className={`wp-image-${mediaId} size-${mediaSizeSlug}${mediaOrientation ? ' is-' + mediaOrientation : ''}`}
                    style={'unset' !== aspectRatio ? {
                        aspectRatio: aspectRatio,
                        objectFit: 'cover',
                        ...(mediaCustomPosition >= 0 && mediaCustomPosition !== 50
                        ? { objectPosition: `${addVerticalPositioning ? '50% ' : ''}${mediaCustomPosition}%${ ! addVerticalPositioning ? ' 50%' : ''}` }
                        : {})
                    } : null }
                />
            ) : (
                <video
                    controls={controls}
                    autoPlay={autoplay}
                    loop={loop}
                    muted={muted}
                    playsInline={playsinline}
                    preload={preload}
                    poster={poster}
                    className={`wp-video-${mediaId}`}
                    style={'unset' !== aspectRatio ? {
                        aspectRatio: aspectRatio,
                        objectFit: 'cover',
                        ...(mediaCustomPosition >= 0 && mediaCustomPosition !== 50
                        ? { objectPosition: `${addVerticalPositioning ? '50% ' : ''}${mediaCustomPosition}%${ ! addVerticalPositioning ? ' 50%' : ''}` }
                        : {})
                    } : null }
                >
                    <source src={mediaUrl} />
                </video>
            )

        ) : (
            <MediaPlaceholder
                onSelect={(media, mediaSizeSlug) => onSelectMedia(media, mediaSizeSlug, setAttributes)}
                allowedTypes={['image', 'video']}
                multiple={false}
                labels={{ title: __('Media area') }}
            />
        )}

        {relatedPostsLinkOption && mediaType && mediaUrl && relatedPosts?.length > 1 ? (<span class="dashicons dashicons-list-view" title="Multiple appearances"></span>) : (<></>)}
        {relatedPostsLinkOption && mediaType && mediaUrl && (! mediaAnchor || mediaAnchor == '-1') ? (<span class="dashicons dashicons-editor-unlink" title="No link attached"></span>) : (<></>)}
        </>        
    );

    return mediaAnchor && mediaAnchor !== '-1'
        ? <figure {...figureProps || {className: `wp-block-${mediaType} wp-block-myconianhotel-media wp-block-myconianhotel-${mediaType} size-${mediaSizeSlug}`}} style={{aspectRatio: aspectRatio}}><a>{figureContent}</a></figure>
        : <figure {...figureProps || {className: `wp-block-${mediaType} wp-block-myconianhotel-media wp-block-myconianhotel-${mediaType} size-${mediaSizeSlug}`}} style={{aspectRatio: aspectRatio}}>{figureContent}</figure>;
}

export const MediaSave = ({
    mediaType,
    mediaUrl,
    mediaAlt,
    mediaId,
    mediaSizeSlug,
    mediaOrientation,
    mediaAnchor,
    relatedPostsLinkOption,
    relatedPosts,
    mediaCustomPosition,
    aspectRatio,
    originalAspectRatio,
    controls,
    autoplay,
    loop,
    muted,
    playsinline,
    preload,
    poster,
    figureProps    
}) => {

    const addVerticalPositioning = verticalPositioning(mediaOrientation, originalAspectRatio, aspectRatio);

    const figureContent = (
        <>
            {mediaType === 'image' && mediaUrl && (
                <img
                    src={mediaUrl}
                    alt={mediaAlt}
                    className={`wp-image-${mediaId} size-${mediaSizeSlug}${mediaOrientation ? ' is-' + mediaOrientation : ''}`}
                    style={'unset' !== aspectRatio ? {
                        aspectRatio: aspectRatio,
                        objectFit: 'cover',
                        ...(mediaCustomPosition >= 0 && mediaCustomPosition !== 50
                            ? { objectPosition: `${addVerticalPositioning ? '50% ' : ''}${mediaCustomPosition}%${ ! addVerticalPositioning ? ' 50%' : ''}` }
                            : {})
                    } : {} }
                />
            )}
            {mediaType === 'video' && mediaUrl && (
                <video
                    controls={controls}
                    autoPlay={autoplay}
                    loop={loop}
                    muted={muted}
                    playsInline={playsinline}
                    preload={preload}
                    poster={poster}
                    className={`wp-video-${mediaId}`}
                    style={'unset' !== aspectRatio ? {
                        aspectRatio: aspectRatio,
                        objectFit: 'cover',
                        ...(mediaCustomPosition >= 0 && mediaCustomPosition !== 50
                            ? { objectPosition: `${addVerticalPositioning ? '50% ' : ''}${mediaCustomPosition}%${ ! addVerticalPositioning ? ' 50%' : ''}` }
                            : {})
                    } : null }
                >
                    <source src={mediaUrl} />
                </video>
            )}
        </>        
    );

    return mediaAnchor && mediaAnchor !== '-1'
        ? <figure {...figureProps || {className: `wp-block-${mediaType} wp-block-myconianhotel-media wp-block-myconianhotel-${mediaType} size-${mediaSizeSlug}`}} style={{aspectRatio: aspectRatio}}><a href={mediaAnchor + (true ? '#wp-image-' + mediaId : '')} rel="nofollow">{figureContent}</a></figure>/*todo*/
        : <figure {...figureProps || {className: `wp-block-${mediaType} wp-block-myconianhotel-media wp-block-myconianhotel-${mediaType} size-${mediaSizeSlug}`}} style={{aspectRatio: aspectRatio}}>{figureContent}</figure>;
}

export const PanelBodyMedia = ({
    mediaType,
    mediaId,
    mediaUrl,
    mediaAlt,
    mediaSizeSlug,
    mediaOrientation,
    mediaAnchor,
    relatedPostsLinkOption,
    relatedPosts,
    mediaCustomPosition,
    aspectRatio,
    originalAspectRatio,
    controls,
    autoplay,
    loop,
    muted,
    playsinline,
    preload,
    poster,
    posterId,
    open,
    setAttributes
}) => {
    const addVerticalPositioning = verticalPositioning(mediaOrientation, originalAspectRatio, aspectRatio);

    const imageSizes = wp.data.select('core/block-editor').getSettings().imageSizes;
    
    const getImageSizeOptions = () => imageSizes.map((size) => ({
        label: size.name,
        value: size.slug,
    }));

    const availableSizes = useSelect(
        (select) => mediaId ? select('core').getMedia(mediaId)?.media_details?.sizes : [],
        [mediaId]
    );

    const onChangeImageSize = (sizeSlug) => {
        if (availableSizes && availableSizes[sizeSlug]) {
            setAttributes({
                mediaSizeSlug: sizeSlug,
                mediaUrl: availableSizes[sizeSlug].source_url,
            });
        }
    };

    const onChangeAltText = (alt) => {
        setAttributes({ mediaAlt: alt });
    };

    const onChangePoster = (posterMedia) => {
        setAttributes({
            poster: posterMedia.url,
            posterId: posterMedia.id
        });
    };

    const onRemovePoster = (setAttributes) => {
        setAttributes({
            poster: '',
            posterId: 0
        });
    };

    //console.log(aspectRatio);

    return (
        <Fragment>
        {mediaType && mediaUrl ? (
            mediaType === 'image' ? (
            <PanelBody title={__('Image Settings')}>
                <TextareaControl
                    label={__('Alternative Text')}
                    value={mediaAlt}
                    onChange={onChangeAltText}
                    help={__('Describe the purpose of the image. Leave empty if the image is purely decorative.')}
                />
                <SelectControl
                    label={__('Image Resolution')}
                    value={mediaSizeSlug}
                    options={getImageSizeOptions()}
                    onChange={onChangeImageSize}
                />
                <SelectControl
                    label="Aspect Ratio"
                    value={aspectRatio}
                    options={[
                        { label: 'Unset', value: 'unset' },                        
                        { label: '1:1', value: '1/1' },                        
                        { label: '2:1', value: '2/1' },                        
                        { label: '3:1', value: '27/10' },                        
                        { label: '3:2', value: '3/2' },                        
                        { label: '4:3', value: '4/3' },                        
                        { label: '16:9', value: '16/9' },
                        { label: '2:3', value: '2/3' },                        
                        { label: '3:4', value: '3/4' },                        
                        { label: '1:2', value: '20/37' }
                    ]}
                    onChange={(newRatio) => {setAttributes({ aspectRatio: newRatio })}}
                />
            </PanelBody>
            ) : (                
            <PanelBody title={__('Video Settings')}>
                <ToggleControl
                    label={__('Autoplay')}
                    checked={autoplay}
                    onChange={() => setAttributes({ autoplay: !autoplay })}
                />
                <ToggleControl
                    label={__('Loop')}
                    checked={loop}
                    onChange={() => setAttributes({ loop: !loop })}
                />
                <ToggleControl
                    label={__('Muted')}
                    checked={muted}
                    onChange={() => setAttributes({ muted: !muted })}
                />
                <ToggleControl
                    label={__('Playback Controls')}
                    checked={controls}
                    onChange={() => setAttributes({ controls: !controls })}
                />
                <ToggleControl
                    label={__('Play Inline')}
                    checked={playsinline}
                    onChange={() => setAttributes({ playsinline: !playsinline })}
                />
                <SelectControl
                    label={__('Preload')}
                    value={preload}
                    options={[
                        { label: 'Auto', value: 'auto' },
                        { label: 'Metadata', value: 'metadata' },
                        { label: 'None', value: 'none' },
                    ]}
                    onChange={(value) => setAttributes({ preload: value })}
                />
                <SelectControl
                    label="Aspect Ratio"
                    value={aspectRatio}
                    options={[
                        { label: 'Unset', value: 'unset' },                        
                        { label: '1:1', value: '1/1' },                        
                        { label: '2:1', value: '2/1' },                        
                        { label: '3:1', value: '27/10' },                        
                        { label: '3:2', value: '3/2' },                        
                        { label: '4:3', value: '4/3' },                        
                        { label: '16:9', value: '16/9' },
                        { label: '2:3', value: '2/3' },                        
                        { label: '3:4', value: '3/4' },                        
                        { label: '1:2', value: '20/37' }
                    ]}
                    onChange={(newRatio) => setAttributes({ aspectRatio: newRatio })}
                />
                <MediaUploadCheck>
                    <MediaUpload
                        onSelect={(posterMedia) => onChangePoster(posterMedia, setAttributes)}
                        allowedTypes={['image']}
                        value={posterId}
                        render={({ open }) => (
                            <Fragment>
                                <Button 
                                    type="button"
                                    aria-describedby="video-block__poster-image-description"
                                    className="components-button is-primary"
                                    onClick={open}
                                >
                                    {poster ? __('Replace Poster Image') : __('Select Poster Image')}
                                </Button>
                                {poster && (
                                    <Button 
                                        type="button"
                                        className="components-button is-secondary"
                                        onClick={() => onRemovePoster(setAttributes)}
                                    >
                                        {__('Remove Poster Image')}
                                    </Button>
                                )}
                            </Fragment>
                        )}
                    />
                </MediaUploadCheck>
            </PanelBody>
            )
        ) : (
            <>{/*No mediaType or mediaUrl*/}</>
        )}
        {mediaType && mediaUrl && relatedPostsLinkOption && (
            <PanelBody title={__(`${mediaType === 'image' ? 'Image' : 'Video'} Link`, 'myconianhotel')}>
                {relatedPosts.length ? (
                <BaseControl
                    help="Link image to a related Page or Post."
                >
                    <SelectControl
                        label=""
                        value={mediaAnchor}
                        options={[{ label: 'None', value: '-1' }, ...relatedPosts]}
                        onChange={(link) => setAttributes({ mediaAnchor: link })}
                    />
                </BaseControl>
                ) : (
                <BaseControl
                    help="No related Page or Post found. Enter a URL manually."
                    className="myconianhotel-linkcontrol-wrapper"
                >
                    <LinkControl
                        key={mediaId} 
                        value={{
                            url: (! mediaAnchor || mediaAnchor == '-1') ? '' : mediaAnchor,
                        }}
                        onChange={(newValue) => {
                            setAttributes({ mediaAnchor: newValue.url });
                        }}
                        withCreateSuggestion={true}
                        forceIsEditingLink={true}
                    />
                    {(mediaAnchor && mediaAnchor !== '-1') && (
                        <Button
                            icon={linkOff}
                            label="Unlink"
                            isDestructive
                            onClick={() => setAttributes({ mediaAnchor: '' })}
                        />
                    )}
                </BaseControl>
                )}
            </PanelBody>
        )}
        {mediaType && mediaUrl && (
            <PanelBody title={__(`${mediaType === 'image' ? 'Image' : 'Video'} Positioning`, 'myconianhotel')}>
                { 'unset' !== aspectRatio ? (
                <Flex>
                    <FlexItem style={{ flexGrow: 1 }}>
                        <RangeControl
                            label={addVerticalPositioning ? __('Vertical (Default: 50%)', 'myconianhotel') :  __('Horizontal (Default: 50%)', 'myconianhotel')}
                            value={mediaCustomPosition}
                            onChange={(value) => setAttributes({ mediaCustomPosition: value })}
                            min={0}
                            max={100}
                        />
                    </FlexItem>
                    <FlexItem>
                        <span>%</span>
                    </FlexItem>
                </Flex>
                ) : (
                    <p>{__('Does not apply when aspect ratio is unset.', 'myconianhotel')}</p>
                )}
            </PanelBody>
        )}
        </Fragment>
    );
}

export const MediaUploadMedia = ({
    mediaId,
    setAttributes
}) => {
    return (
        <MediaUpload
            onSelect={(media, mediaSizeSlug) => onSelectMedia(media, mediaSizeSlug, setAttributes)}
            allowedTypes={['image', 'video']}
            value={mediaId}
            render={({ open }) => (
                <Button onClick={open}>
                    Replace
                </Button>
            )}
        />
    );
}
