const { registerBlockType } = wp.blocks;
const { useBlockProps, useInnerBlocksProps, InspectorControls, InnerBlocks } = wp.blockEditor;
const { PanelBody, SelectControl, TextControl } = wp.components;
const { __ } = wp.i18n;
const { useEffect, useRef } = wp.element;
const { select, dispatch, useSelect } = wp.data;
import { IconSet } from '../../modules';

registerBlockType('myconianhotel/widget-text', {
    apiVersion: 2,
    title: __('Widget & text', 'myconianhotel'),
    category: 'widgets',
    icon: IconSet.widgetText,
    supports: {
        inserter: true,
    },
    attributes: {
        chosenTemplate: {
            type: 'string',
            default: 'position',
        },
        chosenTerm: {
            type: 'string'
        },
        linkText: {
            type: 'string',
            default: 'view'
        }
    },
    edit({ attributes, setAttributes, clientId }) {
        const { chosenTemplate, chosenTerm, linkText } = attributes;
        
        // Initialize term
        useEffect(() => {
            if ('document' == chosenTemplate && ! chosenTerm && terms?.length) {
                setAttributes({chosenTerm: terms[0].id})
            } else if ('document' !== chosenTemplate) {
                setAttributes({chosenTerm: null})
            }
        }, [chosenTemplate])

        // Fetch "position" post titles dynamically
        let posts = useSelect((select) => {
            return select('core').getEntityRecords('postType', chosenTemplate, {
                per_page: -1,
                status: ['publish', 'draft'],
                orderby: 'menu_order',
                order: 'asc',
                'document_type': [chosenTerm ?? null]
            }) || [];
        }, [chosenTemplate, chosenTerm]);

        // Term options
        const terms = useSelect( ( select ) => {
            return select( 'core' ).getEntityRecords('taxonomy', 'document_type', {
                per_page: -1,
            } );
        }, [chosenTemplate]);

        const termOptions = terms?.map((t) => {
            return { label: t.name, value: t.id }
        });

        // Track previous post count
        let lastPostCountRef = useRef(posts?.length || 0);

        // Convert posts to <li> elements for core/list
        //let listItems = posts.map(post => `<li>${post.status === 'publish' ? `<a href="#">` : '' }${post.title.rendered}${post.status === 'publish' ? `<span>${linkText}</span></a>` : '' }</li>`).join('');
        let listItems = posts.map(post => `<li>${post.status === 'publish' && (post.acf?.myconianhotel_document?.file || post.acf?.myconianhotel_document?.url) ? `<a href="#">` : '' }${post.title.rendered}${post.status === 'publish' && (post.acf?.myconianhotel_document?.file || post.acf?.myconianhotel_document?.url) ? `<span>${linkText}</span></a>` : '' }</li>`).join('');
        
        // Define the templates
        const templates = {
            WPForm: [
                ['myconianhotel/free-text', { backgroundColor: '', isNested: true, displayPresets: ['free', 'presetMediaText'], chosenTemplate: 'presetMediaText' }],
                ['wpforms/form-selector', { className: 'wp-block' }]
            ],
            position: [
                ['myconianhotel/free-text', { backgroundColor: '', isNested: true, displayPresets: ['free', 'presetMediaText'], chosenTemplate: 'presetMediaText' }],
                ['core/list', { values: listItems || '<li>No Positions found.</li>', className: 'wp-block is-style-lined' }]
            ],
            event: [
                ['myconianhotel/free-text', { backgroundColor: '', isNested: true, displayPresets: ['free', 'presetMediaText'], chosenTemplate: 'presetMediaText' }],
                ['core/list', { values: listItems || '<li>No Events found.</li>', className: 'wp-block is-style-lined' }]
            ],
            menu: [
                ['myconianhotel/free-text', { backgroundColor: '', isNested: true, displayPresets: ['free', 'presetMediaText'], chosenTemplate: 'presetMediaText' }],
                ['core/list', { values: listItems || '<li>No Menus found.</li>', className: 'wp-block is-style-lined' }]
            ],
            document: [
                ['myconianhotel/free-text', { backgroundColor: '', isNested: true, displayPresets: ['free', 'presetMediaText'], chosenTemplate: 'presetMediaText' }],
                ['core/list', { values: listItems || '<li>No Documents found.</li>', className: 'wp-block is-style-lined' }]
            ]
        };

        // Snipet to update existing blocks
        // Change presets to ['free', 'presetMediaText'] for existing Widget text
        useEffect(() => {
            const innerBlocks = select('core/block-editor').getBlocks(clientId);
            const freeTextBlock = innerBlocks.find(block => block.name === 'myconianhotel/free-text');

            if (
                freeTextBlock && !freeTextBlock.attributes.displayPresets.includes('presetMediaText')
            ) {
                dispatch('core/block-editor').updateBlockAttributes(freeTextBlock.clientId, {
                    displayPresets: ['free', 'presetMediaText'],
                    isNested: true
                });
            }
        }, [clientId]);

        // Track the last chosen template
        const lastTemplateRef = useRef(chosenTemplate);
        const lastLinkTextRef = useRef(linkText);
        const lastTermRef = useRef(chosenTerm);

        // Check existing inner blocks
        const existingBlocks = useSelect((select) => select('core/block-editor').getBlocks(clientId), [clientId]);

        // Update template when switching templates **only if empty**
        useEffect(() => {
            const { replaceInnerBlocks, updateBlockAttributes } = wp.data.dispatch('core/block-editor');

            // If the template has changed, replace all blocks
            if ('WPForm' == chosenTemplate && lastTemplateRef.current !== chosenTemplate) {
                replaceInnerBlocks(clientId, wp.blocks.createBlocksFromInnerBlocksTemplate(templates[chosenTemplate]));
            }
            // If new positions are detected, update only the `core/list` block
            else if (lastTemplateRef.current !== chosenTemplate || lastLinkTextRef.current !== linkText || lastTermRef.current !== chosenTerm || posts?.length !== lastPostCountRef.current) {
                const listBlock = existingBlocks.find(block => block.name === 'core/list');
                
                if (listBlock) {
                    updateBlockAttributes(listBlock.clientId, { values: listItems });
                }
            }
            
            lastPostCountRef.current = posts.length;
            lastTemplateRef.current = chosenTemplate;
            lastLinkTextRef.current = linkText;
            lastTermRef.current = chosenTerm;

        }, [chosenTemplate, chosenTerm, clientId, posts.length, linkText]);

        const blockProps = useBlockProps({
            className: [
                'wp-block is-stacked-on-mobile',
                'alignfull is-layout-constrained has-global-padding',
                'WPForm' !== chosenTemplate ? 'has-lightergray-background-color has-background-color' : 'has-default-background-color has-background-color'
            ].filter(Boolean).join(' ') // Filter out empty class values
        });

        const innerBlocksProps = useInnerBlocksProps({}, {
            template: templates[chosenTemplate],
            templateLock: false
        });

        return (
            <>
                <InspectorControls>
                    <PanelBody title={__('Template Settings', 'myconianhotel')} initialOpen={true}>
                        <SelectControl
                            label={__('Select Template', 'myconianhotel')}
                            value={chosenTemplate}
                            options={[
                                { label: __('WPForm', 'myconianhotel'), value: 'WPForm' },
                                { label: __('Positions List', 'myconianhotel'), value: 'position' },
                                { label: __('Events List', 'myconianhotel'), value: 'event' },
                                { label: __('Menus List', 'myconianhotel'), value: 'menu' },
                                { label: __('Document List', 'myconianhotel'), value: 'document' },
                            ]}
                            onChange={(newTemplate) => setAttributes({ chosenTemplate: newTemplate })}
                        />
                        { 'document' == chosenTemplate && (
                        
                        <SelectControl
                            label={__('Select Document\'s Type', 'myconianhotel')}
                            value={chosenTerm}
                            options={termOptions}
                            onChange={(newTerm) => setAttributes({ chosenTerm: newTerm})}
                        />

                        )}

                        <TextControl
                            label="Link Text"
                            value={linkText}
                            onChange={(value) => setAttributes({ linkText: value })}
                        />
                    </PanelBody>
                </InspectorControls>

                <div {...blockProps}>
                    <div {...innerBlocksProps} />
                </div>
            </>
        );
    },
    save(props) {
        const { attributes } = props;
        const { chosenTemplate } = attributes;

        const blockProps = useBlockProps.save({
            className: [
                'wp-block is-stacked-on-mobile',
                'alignfull is-layout-constrained has-global-padding',
                'WPForm' !== chosenTemplate ? 'has-lightergray-background-color has-background-color' : 'has-default-background-color has-background-color'
            ].filter(Boolean).join(' ') // Filter out empty class values
        });
        const innerBlocksProps = useInnerBlocksProps.save({});

        return (
            <div {...blockProps}>
                <div {...innerBlocksProps} />
            </div>
        );
    }
});

// Register Block Style
wp.blocks.registerBlockStyle('myconianhotel/widget-text', [
    {
        name: 'sticky',
        label: 'Sticky',
    }
]);
