<?php

/**
 * Add Upload column to the Media Library
 * 
 * Make the column sortable
 * 	Modify query to enable sorting by meta key
 * 	Force correct ordering
 * 
 * Add the data (checkbox)
 * 
 * Save the data using AJAX
 * Handle AJAX server-side
 */

/**
 * Add custom column to Media Library
 */
add_filter('manage_upload_columns', function($columns) {
    $columns['myconianhotel_media_gallery_featured'] = __('Gallery page', 'myconianhotel');
    return $columns;
});

/**
 * Make the column sortable
 */
add_filter('manage_upload_sortable_columns', function($sortable_columns) {
    $sortable_columns['myconianhotel_media_gallery_featured'] = 'myconianhotel_featured_attachment';
    return $sortable_columns;
});

/**
 * Modify query to enable sorting by meta key
 */
add_action('pre_get_posts', function($query) {
    if (!is_admin() || !$query->is_main_query()) return;

    $screen = get_current_screen();
    if ($screen && $screen->id === 'upload' && $query->get('orderby') === 'myconianhotel_featured_attachment') {
        $query->set('orderby', 'myconianhotel_featured_attachment_custom');
        add_filter('posts_clauses', 'myconianhotel_featured_attachment_orderby_clause', 10, 2);
    }
});

/**
 * Force correct ordering
 */
function myconianhotel_featured_attachment_orderby_clause($clauses, $query) {
    global $wpdb;

    // Join postmeta table if not already joined
    if (strpos($clauses['join'], 'myconianhotel_featured_attachment') === false) {
        $clauses['join'] .= " LEFT JOIN {$wpdb->postmeta} AS myc_feat_meta 
            ON ({$wpdb->posts}.ID = myc_feat_meta.post_id 
            AND myc_feat_meta.meta_key = 'myconianhotel_featured_attachment')";
    }

    // Sort by meta_value (cast to integer), 1 first, then fallback by post ID
    $clauses['orderby'] = "CAST(myc_feat_meta.meta_value AS UNSIGNED) DESC, {$wpdb->posts}.ID DESC";

    // Remove filter after it's used
    remove_filter('posts_clauses', 'myconianhotel_featured_attachment_orderby_clause', 10);

    return $clauses;
}

/**
 * Render the checkbox in the custom column
 */
add_action('manage_media_custom_column', function($column_name, $post_id) {
    if ($column_name === 'myconianhotel_media_gallery_featured') {
        $checked = get_post_meta($post_id, 'myconianhotel_featured_attachment', true) === '1' ? 'checked' : '';
        echo '<p><input type="checkbox" class="myconianhotel_featured_attachment_checkbox" data-id="' . esc_attr($post_id) . '" ' . $checked . ' /></p>';

        $used_in = find_posts_using_image($post_id);
	    if ($used_in) {
	    	?>
	    	<p>Select image link for Gallery</p>
	    	<select>
	    		<option value="0">None</option>
	    	<?php
	        foreach ($used_in as $post) {
	        	?><option value="<?php echo get_edit_post_link($post->ID); ?>"><?php echo esc_html($post->post_title); ?></option><?php
	        }	        
	    	?>	    		
	    	</select>
	    	<?php
	    } else {
	        echo '<em>Not used</em>';
	    }
    }
}, 10, 2);

/**
 * AJAX script to handle checkbox updates
 */
add_action('admin_footer-upload.php', function() {
    ?>
    <script>
        jQuery(document).ready(function($) {
            $('.myconianhotel_featured_attachment_checkbox').on('change', function() {
                const attachmentId = $(this).data('id');
                const checked = $(this).is(':checked') ? '1' : '0';

                $.post(ajaxurl, {
                    action: 'save_myconianhotel_featured_attachment',
                    attachment_id: attachmentId,
                    value: checked,
                    _ajax_nonce: '<?php echo wp_create_nonce('myconianhotel_featured_attachment_nonce'); ?>'
                });
            });
        });
    </script>
    <?php
});

/**
 * Handle the AJAX request server-side
 */
add_action('wp_ajax_save_myconianhotel_featured_attachment', function() {
    if (
        !current_user_can('upload_files') ||
        !check_ajax_referer('myconianhotel_featured_attachment_nonce', false, false)
    ) {
        wp_send_json_error('Unauthorized');
    }

    $attachment_id = absint($_POST['attachment_id'] ?? 0);
    $value = $_POST['value'] === '1' ? '1' : '0';

    update_post_meta($attachment_id, 'myconianhotel_featured_attachment', $value);
    wp_send_json_success('Updated');
});