<?php

// https://chatgpt.com/c/673b3eda-c2ac-8003-a481-6ffba15cf7e2

/**
 * Disable Gutenberg for accommodation post type
 * 
 * Custom post type Accommodation
 *
 * 404 if term in URL is invalid fix
 * 
 * Accomodation Type taxonomy
 *  Replace %accommodation_type% with the actual term slug in permalinks
 * 
 * Accomodation Amenities taxonomy
 * Save Accomodation Type Taxonomy action
 *  Attach Page to related Taxonomy
 * 
 * Add custom columns to the Accommodation edit screen
 *  Acf fields column
 * Add the data to custom columns for the Accommodation edit screen
 * 
 * Default accommodation template
 * 
 * Filter the excerpt for archive (Tmp disabled)
 * 
 * Attach gallery images to post (optional)
 */

/**
 * Disable Gutenberg for custom post type
 */
//add_filter( 'use_block_editor_for_post_type', 'theme_disable_gutenberg_for_custom_post_type', 10, 2 );
function theme_disable_gutenberg_for_custom_post_type( $current_status, $post_type ) {
    if ( $post_type === 'accommodation' ) return false;
    return $current_status;
}

/**
 * Custom post type Accommodation
 */
add_action('init', 'create_accommodation_post_type');
function create_accommodation_post_type() {
    $labels = array(
        'name'                  => __('Accommodations'),
        'singular_name'         => __('Accommodation'),
        'menu_name'             => __('Accommodations'),
        'name_admin_bar'        => __('Accommodation'),
        'add_new'               => __('Add New'),
        'add_new_item'          => __('Add New'),
        'new_item'              => __('New Accommodation'),
        'edit_item'             => __('Edit Accommodation'),
        'view_item'             => __('View Accommodation'),
        'all_items'             => __('All Accommodations'),
        'search_items'          => __('Search Accommodations'),
        'not_found'             => __('No accommodations found.'),
        'not_found_in_trash'    => __('No accommodations found in Trash.'),
    );

    $args = array(
        'labels'                => $labels,
        'public'                => true,
        'publicly_queryable'    => true,
        'show_ui'               => true,
        'show_in_menu'          => true,
        'query_var'             => true,
        'rewrite' => array(
            'slug'          => 'mykonos-accommodation/%accommodation_type%',
            'with_front'    => false
        ),
        'capability_type'       => 'post',
        'has_archive'           => 'mykonos-accommodation', // ← keeps /accommodation/ as the archive URL
        'hierarchical'          => false,
        'menu_position'         => 5,
        'supports'              => array('title', 'editor', 'thumbnail', 'custom-fields', 'excerpt'),
        'show_in_rest'          => true,  // Enables the Gutenberg editor
        'menu_icon'             => 'dashicons-admin-home'
    );

    register_post_type('accommodation', $args);
}

/**
 * 404 if term in URL is invalid fix
 */
add_action('template_redirect', function () {
    if (is_singular('accommodation')) {
        $term = get_query_var('accommodation_type');
        if ($term && !term_exists($term, 'accommodation_type')) {
            global $wp_query;
            $wp_query->set_404();
            status_header(404);
            nocache_headers();
            include get_query_template('404');
            exit;
        }
    }
});

/**
 * Accomodation Type taxonomy
 */
add_action('init', 'create_accommodation_type_taxonomy');
function create_accommodation_type_taxonomy() {
    $labels = array(
        'name'                  => _x('Types', 'taxonomy general name'),
        'singular_name'         => _x('Type', 'taxonomy singular name'),
        'search_items'          => __('Search Types'),
        'all_items'             => __('All Types'),
        'parent_item'           => __('Parent Type'),
        'parent_item_colon'     => __('Parent Type:'),
        'edit_item'             => __('Edit Type'),
        'update_item'           => __('Update Type'),
        'add_new_item'          => __('Add New Accomodation Type'),
        'new_item_name'         => __('New Type Name'),
        'menu_name'             => __('Types'),
    );

    $args = array(
        'hierarchical'          => true, // True for categories, false for tags
        'labels'                => $labels,
        'show_ui'               => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'rewrite'               => array(
            'slug'          => 'mykonos-accommodation',
            'with_front'    => false
        ),
        'public'                => true,
        'show_in_rest'          => true,  // Enables Gutenberg editor support
    );

    register_taxonomy('accommodation_type', array('accommodation'), $args);
}

/**
 * Replace %accommodation_type% with the actual term slug in permalinks
 */
add_filter('post_type_link', 'custom_accommodation_permalink', 10, 2);
function custom_accommodation_permalink($post_link, $post) {
    if ($post->post_type === 'accommodation') {
        $terms = get_the_terms($post->ID, 'accommodation_type');
        if (!empty($terms) && !is_wp_error($terms)) {
            $term_slug = $terms[0]->slug;
        } else {
            $term_slug = 'uncategorized';
        }

        $post_link = str_replace('%accommodation_type%', $term_slug, $post_link);
    }
    return $post_link;
}

/**
 * Amenities Type taxonomy
 */
add_action('init', 'create_accommodation_amenities_taxonomy');
function create_accommodation_amenities_taxonomy() {
    $labels = array(
        'name'                  => _x('Amenities', 'taxonomy general name'),
        'singular_name'         => _x('Amenity', 'taxonomy singular name'),
        'search_items'          => __('Search Amenities'),
        'all_items'             => __('All Amenities'),
        'parent_item'           => __('Parent Amenity'),
        'parent_item_colon'     => __('Parent Amenity:'),
        'edit_item'             => __('Edit Amenity'),
        'update_item'           => __('Update Amenity'),
        'add_new_item'          => __('Add New Accomodation Amenity'),
        'new_item_name'         => __('New Amenity Name'),
        'menu_name'             => __('Amenities'),
    );

    $args = array(
        'hierarchical'          => true, // True for categories, false for tags
        'publicly_queryable'    => false,
        'public'                => false,
        'rewrite'               => false,
        'labels'                => $labels,
        'show_ui'               => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'show_in_rest'          => true,  // Enables Gutenberg editor support
    );

    register_taxonomy('accommodation_amenities', array('accommodation'), $args);
}

/**
 * Benefits Type taxonomy
 */
add_action('init', 'create_accommodation_benefits_taxonomy');
function create_accommodation_benefits_taxonomy() {
    $labels = array(
        'name'                  => _x('Benefits', 'taxonomy general name'),
        'singular_name'         => _x('Benefit', 'taxonomy singular name'),
        'search_items'          => __('Search Benefits'),
        'all_items'             => __('All Benefits'),
        'parent_item'           => __('Parent Benefit'),
        'parent_item_colon'     => __('Parent Benefit:'),
        'edit_item'             => __('Edit Benefit'),
        'update_item'           => __('Update Benefit'),
        'add_new_item'          => __('Add New Accomodation Benefit'),
        'new_item_name'         => __('New Benefit Name'),
        'menu_name'             => __('Benefits'),
    );

    $args = array(
        'hierarchical'          => true, // True for categories, false for tags
        'labels'                => $labels,
        'publicly_queryable'    => false,
        'public'                => false,
        'rewrite'               => false,
        'show_ui'               => true,
        'show_admin_column'     => true,
        'query_var'             => true,
        'show_in_rest'          => true,  // Enables Gutenberg editor support
    );

    register_taxonomy('accommodation_benefits', array('accommodation'), $args);
}

/**
 * Save Accomodation Type Taxonomy action
 */
add_action('saved_accommodation_type', 'myconianhotel_saved_accommodation_type', 10, 3);
add_action('edited_accommodation_type', 'myconianhotel_saved_accommodation_type', 10, 3);
function myconianhotel_saved_accommodation_type( $term_id, $tt_id, $args ) {

    if ( $args['acf'] ) {
        // Check for the correct field
        foreach ( $args['acf'] as $key => $related_page ) {
            $field = acf_maybe_get_field( $key );
            if ( 'myconianhotel_related_page' == ( $field['name'] ?? false ) ) {

                // Get all pages with myconianhotel_related_object meta
                $pages = get_pages( array( 'meta_key' => 'myconianhotel_related_object' ) );
                if ( $pages ) {
                    foreach ( $pages as $page ) {
                        $meta = get_post_meta( $page->ID, 'myconianhotel_related_object', true );
                        // Remove the key if it exists
                        if ( isset( $meta[$term_id] ) && $term_id ) {                       
                            unset( $meta[$term_id] );
                            // Update the meta with the new array
                            /// if ( empty( $meta ) ) { $meta = ''; }
                            update_post_meta( $page->ID, 'myconianhotel_related_object', $meta );
                            break;
                        }
                    }
                }

                if ( $related_page ) {
                    $term = get_term( $term_id );
                    $taxonomy_details = get_taxonomy( $term->taxonomy );
                    $post_type = get_post_type_object( $taxonomy_details->object_type[0] ); // TODO
                    // Get the existing meta value
                    $meta = get_post_meta( $related_page, 'myconianhotel_related_object', true ) ?: array();
                    // Add or update the key with related page value
                    $meta[$term_id] = array(
                        'title' => $post_type->labels->name . ' / ' . $taxonomy_details->labels->name . ' / ' . $term->name,
                        'link' => get_edit_term_link( $term_id ),
                        'ID' => $term_id
                    );
                }

                if ( $related_page && ! empty( $meta ) ) {
                    // Update the meta with the new array
                    update_post_meta( $related_page, 'myconianhotel_related_object', $meta );
                }
            }
        }
    }
}


/**
 * Add custom columns to the Accommodation edit screen
 */
add_filter( 'manage_accommodation_posts_columns', function( $columns ) {
    $new_columns = array();

    foreach ($columns as $key => $value) {
        $new_columns[$key] = $value;

        // Insert navigation column after the Title column.
        if ('title' === $key) {
            $new_columns['post_acf_fields'] = __('Attributes', 'myconianhotel');
            ?>
            <style>
                .column-post_acf_fields ul {
                    margin: 0;
                    display: flex;
                    flex-wrap: wrap;
                }

                .column-post_acf_fields ul li:not(:last-child):after {
                    content: ',\00a0';
                }
            </style>
            <?php
        }
    }

    return $new_columns;
});

/**
 * Add the data to custom columns for the Accommodation edit screen
 */
add_action( 'manage_accommodation_posts_custom_column', function( $column, $post_id ) {
    if ($column === 'post_acf_fields') {
        get_template_part( '/theme-blocks/blocks/post-acf-fields/template', '', array(
            'post_id' => $post_id,
            'filter' => 'is_featured',
            'title' => false
        ));
    }
}, 10, 2);


/**
 * Default accommodation template
 */
add_filter( 'block_editor_settings_all', 'myconianhotel_add_default_block_template_for_accommodation', 10, 2 );
function myconianhotel_add_default_block_template_for_accommodation( $editor_settings, $post ) {
    // Apply only for new posts and a specific post type, e.g., 'post'.
    if ( $post->post ) {
        if ($post->post->post_type === 'accommodation' && $post->post->post_status === 'auto-draft') {
            // Define the block template.
            $editor_settings['template'] = [
                ['myconianhotel/gallery', [
                    'backgroundColor' => 'transparent',
                    'layout' => '',
                    'aspectRatio'=> 'unset',
                    'sliderTheme' => 'autowidth',
                    'className' => 'is-style-alternative'
                ]],
                ['myconianhotel/post-info'],            
                ['myconianhotel/post-navigation'],
                ['myconianhotel/media', ['align' => 'full', 'aspectRatio' => '27/10']]
            ];

            // Optionally lock the template to prevent adding/removing blocks.
            $editor_settings['templateLock'] = 'all'; // Set to 'all' to lock completely.
        }        
    }
    return $editor_settings;
}

/*add_filter('excerpt_allowed_blocks', function ($allowed_blocks) {
    // Add your custom block to the allowed blocks for excerpts.
    $allowed_blocks[] = 'myconianhotel/paragraph';
    return $allowed_blocks;
});*/


/**
 * Filter the excerpt for archive (Tmp disabled)
 */
add_filter( 'get_the_excerpt', function ( $excerpt, $post ) {

    if ( $post->post_type !== 'accommodation' ) { return $excerpt; }

    if ( empty( $excerpt ) ) {
        // Parse the post content into blocks.
        $blocks = parse_blocks( $post->post_content );

        // Get the excerpt from the inner blocks.
        $excerpt = get_myconian_paragraph_excerpt( $blocks, 50 );
    }
    return $excerpt;
}, 10, 2 );

// Tmp disabled
function get_myconian_paragraph_excerpt( $blocks, $word_count = 20 ) { // 20 words by default.
    foreach ( $blocks as $block ) {
        // If this is the 'myconianhotel/paragraph' block, extract its content.
        if ( $block['blockName'] === 'myconianhotel/paragraph' && ! empty( $block['innerHTML'] ) ) {
            $raw_content = strip_tags( $block['innerHTML'] ); // Remove HTML tags.
            // Tmp disabled return wp_trim_words( $raw_content, $word_count ); // Return trimmed content.
            return $raw_content;
        }

        // If this block contains inner blocks, recursively search them.
        if ( ! empty( $block['innerBlocks'] ) ) {
            $inner_excerpt = get_myconian_paragraph_excerpt( $block['innerBlocks'], $word_count );
            if ( $inner_excerpt ) {
                return $inner_excerpt; // Return the excerpt if found.
            }
        }
    }

    // Return empty if no excerpt is found.
    return '';
}

/**
 * Attach gallery images to post (optional)
 */
//add_action( 'save_post', 'myconianhotel_attach_gallery_images_to_accommodation' );
function myconianhotel_attach_gallery_images_to_accommodation( $post_id ) {
    // Exit if this is an autosave or a revision.
    if ( wp_is_post_autosave( $post_id ) || wp_is_post_revision( $post_id ) ) {
        return;
    }

    // Get the post content.
    $post_content = get_post_field( 'post_content', $post_id );

    // Check if the post contains a gallery block.
    if ( has_block('myconianhotel/gallery', $post_content ) ) {
        // Parse the blocks in the post content.
        $blocks = parse_blocks($post_content);

        foreach ($blocks as $block) {
            if ( $block['blockName'] === 'myconianhotel/gallery' ) {
                // Get the IDs of images in the gallery block.
                $image_ids = array();
                foreach ( $block['innerBlocks'] as $block ) {
                    $image_ids[] = $block['attrs']['mediaId'];
                }

                foreach ( $image_ids as $image_id ) {
                    // Update the post_parent of each image to the current post.
                    wp_update_post([
                        'ID'          => $image_id,
                        'post_parent' => $post_id,
                    ]);
                }
            }
        }
    }
}