<?php

/**
 * Customize WPForms
 * 
 * Register the 'medium_large' size
 * Update medium_large core image size
 * 
 * Extract myconianhotels shortcodes from text
 * myconianhotels link shortcode
 * Check if is localhost
 * Get Archive related Page
 * Check if has Hero block
 * Default Page template
 * 
 * Modify Archives query order
 * Pages edit screen states (404 Page)
 *
 * ACF
 *  Save Option page action
 *      Attach Page to related Option page
 * 
 *  Modify HTML for Option page before (disabled)
 *  Modify HTML for Option page after (disabled)
 * 
 *  Update ACF taxonomy related location rules
 *      Page
 *      Hero block
 *      Free text block
 *      Related Page
 * 
 *  Admin head scripts (disabled)
 *      Metaboxes always closed
 *  
 * Helpers
 *  String replace last occurance
 *  Insert value in assosiative array after key
 *  Find posts using image
 *  Get Myconian Collection remote data
 *  Timestamp to date
 *  Clean svg markup
 */

// Customize WPForms
require_once __DIR__ . '/WPForms-customize.php';
//require_once __DIR__ . '/WP-media-library-customize.php';

/**
 * Register the 'medium_large' size
 */
function theme_add_image_size_to_editor( $sizes ) {
    $custom_sizes = [];
    
    foreach ( $sizes as $key => $label ) {
        $custom_sizes[$key] = $label;
        
        // Check for the 'medium' size and add 'medium_large' right after it
        if ( 'medium' === $key ) {
            $custom_sizes['medium_large'] = __( 'Medium Large' );
        }
    }

    return $custom_sizes;
}

/**
 * Update medium_large core image size
 */
function theme_update_medium_large_image_size() {
    // Get the current image sizes
    $image_sizes = get_intermediate_image_sizes();
    // Check if "medium_large" is among the registered image sizes
    if ( in_array( 'medium_large', $image_sizes ) ) {
        // Get the current sizes of "medium_large"
        $medium_large_size = image_get_intermediate_size( 'medium_large', 0 );
        if ($medium_large_size) {
           // Update the dimensions of "medium_large"
            $new_width = ''; // New width in pixels
            $new_height = 1080; // New height in pixels
            $medium_large_size['width'] = $new_width;
            $medium_large_size['height'] = $new_height;

            // Update the "medium_large" size settings
            update_option( 'medium_large_size_w', $new_width );
            update_option( 'medium_large_size_h', $new_height );
        }
        
    }
}

/**
 * Extract myconianhotels shortcodes from text
 */
function myconianhotel_shortcodes( $text ) {
    $shortcode = '';
    preg_match('/\[myconianhotel(.*?)\]/', $text, $matches);

    if (!empty($matches[1])) {
        $extracted_text = trim( $matches[1] );
        $shortcode = '[myconianhotel ' . $extracted_text . ']';
    }

    $shortcode_result = do_shortcode( $shortcode );

    return str_replace( $shortcode, $shortcode_result, $text );
}


/**
 * myconianhotels link shortcode
 */
add_shortcode( 'myconianhotel', 'myconianhotel_link_shortcode' );
function myconianhotel_link_shortcode($atts) {
    $atts = shortcode_atts(
        [
            'url' => '#',
            'title' => 'Link'
        ],
        $atts,
        'myconianhotel'
    );

    // Return the anchor tag
    return sprintf(
        '<a href="%s" title="%s" target="_BLANK">%s</a>',
        esc_url($atts['url']),
        esc_attr($atts['title']),
        esc_html($atts['title'])
    );
}

/**
 * Check if is localhost
 */
function myconianhotel_is_localhost() {
    // Define localhost values
    $whitelist = ['127.0.0.1', '::1', 'localhost'];
    // Check if the current host is in the whitelist
    return in_array($_SERVER['REMOTE_ADDR'], $whitelist) || in_array($_SERVER['SERVER_NAME'], $whitelist);
}

/**
 * Get Archive related Page
 */
function myconianhotel_archive_related_page( $post_type, $taxonomy, $options_page ) {
    // Related Page
    if ( is_tax() ) { // Category
        // Texonomy's related Page
        return get_field( 'myconianhotel_related_page', $taxonomy . '_' . get_queried_object_id() );
    } else if ( is_archive() ) { // Uncategorized archive
        // Archives's related Page
        return get_field( 'myconianhotel_related_page', $options_page );
    }
}

/**
 * Check if has Hero block
 */
function myconian_hotel_has_hero_block( $archive_related_page, $options_page ) {
    if (
        has_block( 'myconianhotel/hero' )
        || ( has_block( 'myconianhotel/hero', $archive_related_page ?? null ) )
        || ( ! $archive_related_page && ( get_field( 'myconianhotel_hero_block_video', $options_page ) || get_field( 'myconianhotel_hero_block_image', $options_page ) ) )
    ) {
        return true;
    } return false;
}

/**
 * Default Page template
 */
add_filter( 'block_editor_settings_all', 'myconianhotel_add_default_block_template_for_page', 10, 2 );
function myconianhotel_add_default_block_template_for_page( $editor_settings, $post ) {
    // Apply only for new posts and a specific post type, e.g., 'post'.
    if ( $post->post ) {
        if ($post->post->post_type === 'page' && $post->post->post_status === 'auto-draft') {
            // Define the block template.
            $editor_settings['template'] = [
                ['myconianhotel/free-text', array('displayPresets' => array('free'))]
            ];
        }        
    }
    return $editor_settings;
}

/**
 * Modify Archives query order
 */
add_action( 'pre_get_posts', 'myconianhotel_modify_archive_query_order' );
function myconianhotel_modify_archive_query_order( $query ) {
    if ( ! is_admin() && $query->is_main_query() && $query->is_post_type_archive('accommodation') ) {
        $query->set( 'orderby', array( 'menu_order' => 'ASC', 'date' => 'DESC' ) ); 
        $query->set( 'order', 'ASC' );
    }
}

/**
 * Pages edit screen states (404 Page)
 */
add_filter( 'display_post_states', function( $states, $post ) {
    $p_404 = get_option( 'myconianhotel_404_page' );

    if ( $p_404 && $post->ID == $p_404 ) {
        $states[] = __( '404 Page' );
    }

    return $states;
}, 10, 2 );

/**
 * ACF
 * Save Option page action
 */
add_action('acf/options_page/save', 'myconianhotel_save_options_page', 10, 2);/////
function myconianhotel_save_options_page( $post_id, $menu_slug ) {
    // Related Page
    $related_page = get_fields( $post_id )['myconianhotel_related_page'] ?? null;

    // Get all Pages with myconianhotel_related_object meta
    $pages = get_pages( array( 'meta_key' => 'myconianhotel_related_object' ) );
    if ( $pages ) {
        foreach ( $pages as $page ) {
            $meta = get_post_meta( $page->ID, 'myconianhotel_related_object', true );
            // Remove the key if it exists
            if ( isset( $meta[$menu_slug] ) ) {
                unset( $meta[$menu_slug] );
                // Update the meta with the new array
                /// if ( empty( $meta ) ) { $meta = ''; }
                update_post_meta( $page->ID, 'myconianhotel_related_object', $meta );
                break;
            }
        }
    }

    if ( $related_page ) {
        $post_type = get_post_type_object( $_GET['post_type'] );
        // Get the existing meta value
        $meta = get_post_meta( $related_page, 'myconianhotel_related_object', true ) ?: array();
        // Add or update the key with related Page value
        $meta[$menu_slug] = array(
            'title' => $post_type->labels->name . ' / Archive page',
            'link' => admin_url() . 'edit.php?post_type=' . $_GET['post_type'] . '&page=' . $_GET['page'],
            'ID' => $post_id
        );
    }
    
    if ( $related_page && ! empty( $meta ) ) {
        // Update the meta with the new array
        update_post_meta( $related_page, 'myconianhotel_related_object', $meta );
    }
}

/**
 * ACF
 * Modify HTML for Option page before (disabled)
 */
add_action('restaurant_page_restaurant-archive-page', 'myconianhotel_before_acf_options_page', 1);
add_action('hotel_page_hotel-archive-page', 'myconianhotel_before_acf_options_page', 1);
add_action('accommodation_page_accommodation-archive-page', 'myconianhotel_before_acf_options_page', 1);
function myconianhotel_before_acf_options_page() {
    // Before ACF outputs the Options page content start an object buffer so that we can capture the output
    ob_start();
}
   
/**
 * ACF
 * Modify HTML for Option page after (disabled)
 */
add_action('restaurant_page_restaurant-archive-page', 'after_acf_options_page', 20);
add_action('hotel_page_hotel-archive-page', 'after_acf_options_page', 20);
add_action('accommodation_page_accommodation-archive-page', 'after_acf_options_page', 20);
function after_acf_options_page() {
    // After ACF finishes get the output and modify it
    $content = ob_get_clean();    
    $count = 1; // the number of times we should replace any string    

    $my_content = '<p><span class="dashicons dashicons-info-outline"></span> Hero & Free text blocks appear in Archive and in any related Category Page.<p>';
    $my_content .= '<p><span class="dashicons dashicons-warning"></span> Hero & Free text blocks are disabled for Archive Page if any <strong>Related Page</strong> is selected. But they still appear in related Category Pages.</p>';
    $content = str_replace('</h1>', '</h1>' . $my_content, $content, $count);    

    echo $content;
}

/**
 * ACF
 * Update ACF taxonomy related location rules
 */
add_filter('acf/get_field_group', 'myconianhotel_field_group_location');
function myconianhotel_field_group_location( $group ) {

    $colorscheme = get_option( 'myconianhotel_colorscheme' );
    $taxonomy = 'collection' == $colorscheme ? 'location' : 'accommodation_type';
    $add_taxonomies = 'collection' == $colorscheme ? array( 'location', 'restaurant_location' ) : array( 'accommodation_type' );
    $updated_group_location = array();

    if (
    'group_67a349678bbe6' == $group['key'] // Page
    || 'group_6745ff77ab68b' == $group['key'] // Hero block
    || 'group_67a206ba9d5fa' == $group['key'] // Free text block
    || 'group_6797a0e026151' == $group['key'] // Related Page
    ) {
        // Remove all taxonomy related params
        foreach ( $group['location'] as $location ) {
            if ( 'taxonomy' !== $location[0]['param'] ) {
                $updated_group_location[] = $location;
            }
        }
        // Add taxonomy related params
        foreach ( $add_taxonomies as $taxonomy ) {
            $updated_group_location[] = array(
                array(
                    "param" => "taxonomy",
                    "operator" => "==",
                    "value" => $taxonomy
                )
            );
        }

        $group['location'] = $updated_group_location;
    }


    return $group;
};


/**
 * ACF
 * Admin head scripts (disabled)
 * Metaboxes always closed
 */
//add_action('acf/input/admin_head', 'my_acf_input_admin_head');
function my_acf_input_admin_head() {
    ?>
    <script type="text/javascript">
    jQuery(function(){
        
      jQuery('.acf-postbox').addClass('closed');
    });
    </script>
<?php
}

/**
 * Helpers
 * String replace last occurance
 */
function myconianhotel_str_lreplace($search, $replace, $subject) {
    $pos = strrpos($subject, $search);

    if ($pos !== false) {
        $subject = substr_replace($subject, $replace, $pos, strlen($search));
    }

    return $subject;
}

/**
 * Helpers
 * Insert value in assosiative array after key
 */
function myconianhotel_array_insert_after( $array, $key, $new ) {
    $keys = array_keys( $array );
    $index = array_search( $key, $keys );
    $pos = false === $index ? count( $array ) : $index + 1;

    return array_merge( array_slice( $array, 0, $pos ), $new, array_slice( $array, $pos ) );
}

/**
 * Helpers
 * Find posts using image
 */
function find_posts_using_image( $attachment_id, $exclude_ids = [] ) {
    $query_args = [
        'post_type'      => ['post', 'page', 'accommodation', 'hotel', 'restaurant', 'event'],
        'posts_per_page' => -1,
        's'              => '"id":' . $attachment_id, // matches block-style image ID usage
    ];

    if (!empty($exclude_ids)) {
        $query_args['post__not_in'] = array_map('intval', $exclude_ids);
    }

    $query = new WP_Query($query_args);

    return $query->posts;
}

/**
 * Helpers
 * Get Myconian Collection remote data
 */
function myconianhotel_remote_data($force_update = false) {
    $myconianhotel_id = get_option( 'myconianhotel_id' );

    if ( $myconianhotel_id && get_option( 'myconianhotel_colorscheme' ) !== 'collection' ) {
        $feed = myconianhotel_fetch_block_data(
            'https://myconiancollection.gr/wp-json/custom/v1/hotels',
            'myconianhotel_hotels_block_data',
            DAY_IN_SECONDS,
            $force_update
        );

        if ( $feed['hotels'] ?? false ){
            $keys = array_column($feed['hotels'], "id"); 
            $index = array_search( $myconianhotel_id, $keys );
            if ( $index !== false ) {
                return $feed['hotels'][$index];
            } 
        }
          
    }
    
    return array();
}

/**
 * Helpers
 * Timestamp to date
 */
function myconianhotel_timestamp_to_date( $timestamp ) {
    if ( $timestamp ) {        
        $timezone = new DateTimeZone( wp_timezone_string() );
        $dt = new DateTime("@$timestamp"); // "@" tells it to interpret as a timestamp
        $dt->setTimezone($timezone); // convert to your timezone     
        return $dt->format('d/M/Y H:i:s') ?: array();
    }

    return array();
}

/**
 * Helpers
 * Clean svg markup
 *  Remove ids and comments
 */
function myconianhotel_clean_svg_markup( $src ) {
    $input = file_get_contents( $src );
    $export = preg_replace( '/<!--(.*?)-->/s', '', $input );
    $export = preg_replace( '/id="(.*?)"/s', '', $export );
    return $export;
}
