import { link } from '@wordpress/icons'; // Import the link icon

const { registerBlockType } = wp.blocks;
const { __ } = wp.i18n;
const { useBlockProps, BlockControls, RichText, __experimentalLinkControl, InspectorControls } = wp.blockEditor;
const { ToolbarButton, Popover, PanelBody, SelectControl } = wp.components;
const { useState, Fragment } = wp.element;
const { displayShortcut } = wp.keycodes;

// Inline SVG icon as a string
const myCustomIcon = (
    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
            <path d="M14.5 17.5H9.5V16H14.5V17.5Z M14.5 8H9.5V6.5H14.5V8Z M7 3.5H17C18.1046 3.5 19 4.39543 19 5.5V9C19 10.1046 18.1046 11 17 11H7C5.89543 11 5 10.1046 5 9V5.5C5 4.39543 5.89543 3.5 7 3.5ZM17 5H7C6.72386 5 6.5 5.22386 6.5 5.5V9C6.5 9.27614 6.72386 9.5 7 9.5H17C17.2761 9.5 17.5 9.27614 17.5 9V5.5C17.5 5.22386 17.2761 5 17 5Z M7 13H17C18.1046 13 19 13.8954 19 15V18.5C19 19.6046 18.1046 20.5 17 20.5H7C5.89543 20.5 5 19.6046 5 18.5V15C5 13.8954 5.89543 13 7 13ZM17 14.5H7C6.72386 14.5 6.5 14.7239 6.5 15V18.5C6.5 18.7761 6.72386 19 7 19H17C17.2761 19 17.5 18.7761 17.5 18.5V15C17.5 14.7239 17.2761 14.5 17 14.5Z"></path>
    </svg>
);

registerBlockType('myconianhotel/buttons', {
    apiVersion: 2,
    title: __('Button (M)', 'myconianhotel'),
    category: 'common',
    icon: 'button',
    supports: {
        anchor: true, // Enables anchor support
        inserter: false, // Prevents this block from appearing in the block inserter
    },
    attributes: {
        url: {
            type: 'string',
            source: 'attribute',
            selector: 'a',
            attribute: 'href',
            default: '',
        },
        linkToForm: {
            type: 'string',
            default: ''
        },
        opensInNewTab: {
            type: 'boolean',
            default: false,
        },
        text: {
            type: 'string',
            source: 'html',
            selector: 'a',
            default: '',
        },
    },
    edit({ attributes, setAttributes, isSelected }) {
        const { url, linkToForm, opensInNewTab, text } = attributes;
        const [isEditingURL, setIsEditingURL] = useState(false); // State for controlling the link editing
        const blockProps = useBlockProps({
            className: 'wp-block-buttons is-layout-flex wp-block-buttons-is-layout-flex', // Add class for the outer wrapper
        });

        const startEditing = () => {
            setIsEditingURL(true);
        };

        const stopEditing = () => {
            setIsEditingURL(false);
        };

        const wpforms = [
            { label: "None", value: "" },
            ...(myconianhotelBlockButtons.wpforms?.map(form => ({
                label: form.post_title,
                value: form.ID
            })) ?? [])
        ];


        return (
            <Fragment>
                {/* Toolbar for link editing */}
                <BlockControls>
                    <ToolbarButton
                        name="link"
                        icon={link}
                        title={__('Edit Link', 'myconianhotel')}
                        shortcut={displayShortcut.primary('k')}
                        onClick={startEditing}
                        style={
                            {backgroundColor: ! url ? 'yellow' : 'transparent' }
                        }
                    />
                </BlockControls>
                <InspectorControls>
                    <PanelBody title="Options">
                        <SelectControl
                            label="Open form modal"
                            value={linkToForm}
                            options={wpforms}
                            onChange={(value) => setAttributes({
                                linkToForm: value,
                                url: value ? '#' : ''
                            })}
                        />
                    </PanelBody>
                </InspectorControls>

                {/* Show the Popover when editing the URL */}
                {isSelected && isEditingURL && (
                    <Popover position="bottom-center" onClose={stopEditing}>
                        <__experimentalLinkControl
                            value={{ url, opensInNewTab }}
                            onChange={({ url: newUrl, opensInNewTab: newTab }) => {
                                setAttributes({ url: newUrl, opensInNewTab: newTab });
                            }}
                            onRemove={() => {
                                setAttributes({ url: '' });
                                stopEditing();
                            }}
                        />
                    </Popover>
                )}

                {/* RichText to allow editing the button text as a div in admin */}
                <div {...blockProps}>
                    <div className={"wp-block-button" + (linkToForm ? ' has-link-to-form' : '')}>
                        <RichText
                            tagName="div" // Use div in the admin
                            className="wp-block-button__link wp-element-button"
                            value={text}
                            onChange={(newText) => setAttributes({ text: newText })}
                            placeholder={__('Add text...', 'myconianhotel')}
                            allowedFormats={[]}
                        />
                    </div>
                        { ! url && false && <mark style={{position:'absolute',top:'100%',marginTop:'1em','fontSize':'.8em',padding:'.2em .5em'}}>Link is missing</mark>}
                </div>
            </Fragment>
        );
    },
    save({ attributes }) {
        const { url, linkToForm, opensInNewTab, text } = attributes;
        const blockProps = useBlockProps.save({
            className: 'wp-block-buttons is-layout-flex wp-block-buttons-is-layout-flex', // Add class for the outer wrapper
        });

        return (
            <Fragment>
            { text && url && (
                <div {...blockProps}>
                    <div className="wp-block-button">
                        <RichText.Content
                            tagName="a" // Use anchor tag in the front end
                            className="wp-block-button__link wp-element-button"
                            href={linkToForm ? linkToForm : (url || undefined)} // Avoid rendering empty href
                            value={text}
                            target={opensInNewTab ? '_blank' : undefined}
                            rel={opensInNewTab ? 'noopener noreferrer' : undefined}
                        />
                    </div>
                </div>
            )}
            </Fragment>
        );
    }
});
