const { registerBlockType } = wp.blocks;
const { InspectorControls, InnerBlocks, useBlockProps, useInnerBlocksProps, useSetting } = wp.blockEditor;
const { PanelBody, SelectControl } = wp.components;
const { useSelect } = wp.data;
const { useState, useEffect, useRef } = wp.element;
const { __ } = wp.i18n;
import { getTextContent } from '@wordpress/rich-text';
import { CustomPanelColorSettings, IconSet } from '../../modules';

registerBlockType('myconianhotel/free-text', {
    apiVersion: 2,
    title: __('Free text', 'myconianhotel'),
    icon: IconSet.freeText,
    category: 'layout',
    supports: {
        /*color: {
            background: true,
        },*/
        inserter: true,
    },
    attributes: {
        isNested: { type: 'bool', default: false },
        chosenTemplate: { type: 'string', default: 'free' },
        displayPresets: { type: 'array', default: ['free', 'presetMediaTextLarge', 'presetMediaTextXxlarge'] },
        backgroundColor: { // supports
            type: 'string'
        }
    },
    edit: ({ attributes, setAttributes, clientId }) => {
        const { chosenTemplate, isNested, displayPresets, backgroundColor } = attributes;
        const allowedBlocksByTemplate = {
            free: ['core/paragraph', 'core/heading', 'core/list', 'core/list-item', 'core/quote', 'core/table'],
            presetMediaText: ['myconianhotel/paragraph', 'myconianhotel/heading', 'myconianhotel/buttons'],
            presetMediaTextLarge: ['myconianhotel/paragraph', 'myconianhotel/heading', 'myconianhotel/buttons'],
            presetMediaTextXxlarge: ['myconianhotel/paragraph', 'myconianhotel/heading', 'myconianhotel/buttons']
        };
        const [allowedBlocks, setAllowedBlocks] = useState(allowedBlocksByTemplate[chosenTemplate]);
        const [lockTemplate, setLockTemplate] = useState(chosenTemplate !== 'free');

        // Define templates and allowed blocks
        const templates = {
            free: [
                ['core/paragraph', {fontSize: 'small', placeholder: __('Paragraph', 'myconianhotel')}],
                ['core/heading', {placeholder: __('Heading', 'myconianhotel')}],
                ['core/paragraph', {placeholder: __('Paragraph', 'myconianhotel')}],
                ['myconianhotel/buttons', {url:'', text:'' }]
            ],
            presetMediaText: [
                ['myconianhotel/paragraph', {text: '', textAlign: ! isNested ? 'center' : '', fontSize: 'small', placeholder: __('Subtitle', 'myconianhotel')}],
                ['myconianhotel/heading', {content: '', level: 2, textAlign: ! isNested ? 'center' : '', placeholder: __('Heading', 'myconianhotel')}],
                ['myconianhotel/paragraph', {text: '', textAlign: ! isNested ? 'center' : '', fontSize: 'medium', placeholder: __('Content', 'myconianhotel')}],
                ['myconianhotel/buttons', {url:'', text:'' }]
            ],
            presetMediaTextLarge: [
                ['myconianhotel/paragraph', {text: '', textAlign: 'center', fontSize: 'small', placeholder: __('Subtitle', 'myconianhotel')}],
                ['myconianhotel/heading', {content: '', level: 2, textAlign: 'center', placeholder: __('Heading', 'myconianhotel')}],
                ['myconianhotel/paragraph', {text: '', textAlign: 'center', fontSize: 'large', placeholder: __('Content', 'myconianhotel')}],
                ['myconianhotel/buttons', {url:'', text:'' }]
            ],
            presetMediaTextXxlarge: [
                ['myconianhotel/paragraph', {text: '', textAlign: 'center', fontSize: 'medium', placeholder: __('Subtitle', 'myconianhotel')}],
                ['myconianhotel/heading', {content: '', level: 2, textAlign: 'center', placeholder: __('Heading', 'myconianhotel')}],
                ['myconianhotel/paragraph', {text: '', textAlign: 'center', fontSize: 'xxlarge', placeholder: __('Content', 'myconianhotel')}],
                ['myconianhotel/buttons', {url:'', text:'' }]
            ]
        };

        const presetsNames = {
            free: __('Free', 'myconianhotel'),
            presetMediaText: __('Media & Text preset', 'myconianhotel'),
            presetMediaTextLarge: __('Media & Text large preset', 'myconianhotel'),
            presetMediaTextXxlarge: __('Media & Text extra large preset', 'myconianhotel')
        };

        const options = displayPresets.map((p) => {
            if (templates[p]) {
                return { label: presetsNames[p], value: p }
            }
        });

        const existingBlocks = useSelect(
            (select) => select('core/block-editor').getBlocks(clientId),
            [clientId]
        );

        // Track last chosen template
        const lastTemplateRef = useRef(chosenTemplate);

        // Switching between templates and keep content if is doable
        useEffect(() => {
            if (lastTemplateRef.current !== chosenTemplate) {
                setAllowedBlocks(allowedBlocksByTemplate[chosenTemplate] || []);
                const { updateBlockAttributes, replaceInnerBlocks } = wp.data.dispatch('core/block-editor');
                let updatedChosenTemplate = templates[chosenTemplate];

                // Switch between solid compatible templates
                if ('free' !== lastTemplateRef.current && 'free' !== chosenTemplate) {
                    existingBlocks.forEach((block, index) => {
                        if ('myconianhotel/paragraph' === block.name) {
                            if (0 === index) {
                                updatedChosenTemplate[0][1].text = block.attributes.text; // Subtitle         
                            } else if ((2 === index || 1 === index)) {
                                updatedChosenTemplate[2][1].text = block.attributes.text;  // Content                                    
                            }
                        }

                        if ('myconianhotel/heading' === block.name && (0 === index || 1 === index)) {
                            updatedChosenTemplate[1][1].content = block.attributes.content; // Heading
                        }
                        
                        if ('myconianhotel/buttons' === block.name) {
                            updatedChosenTemplate[3][1].url = block.attributes.url;
                            updatedChosenTemplate[3][1].text = block.attributes.text;
                        }
                    });
                }
                // Switch between free and any preset template
                else if ( 'free' == lastTemplateRef.current ) {
                    existingBlocks.forEach((block, index) => {                            
                        if ('core/paragraph' === block.name) {
                            if (0 === index) {
                                updatedChosenTemplate[0][1].text = getTextContent(block.attributes.content); // Subtitle                                    
                            } else if ((2 === index || 1 === index) /*|| (0 === index && 0 == existingBlocks.length)*/) {
                                updatedChosenTemplate[2][1].text = getTextContent(block.attributes.content);  // Content
                            }
                        }

                        if ('core/heading' === block.name && (0 === index || 1 === index)) {
                            updatedChosenTemplate[1][1].content = getTextContent(block.attributes.content); // Heading
                        }
                        
                        if ('myconianhotel/buttons' === block.name && (2 === index || 3 === index)) {
                            updatedChosenTemplate[3][1].url = block.attributes.url;
                            updatedChosenTemplate[3][1].text = block.attributes.text;
                        }
                    });
                }
                // Switch between any preset template and free
                // TODO: is the same with first contition except .text is .content
                else if ('free' == chosenTemplate) {
                    existingBlocks.forEach((block, index) => {
                        if ('myconianhotel/paragraph' === block.name) {
                            if (0 === index) {
                                updatedChosenTemplate[0][1].content = block.attributes.text; // Subtitle         
                            } else if ((2 === index || 1 === index)) {
                                updatedChosenTemplate[2][1].content = block.attributes.text;  // Content                                    
                            }
                        }

                        if ('myconianhotel/heading' === block.name && (0 === index || 1 === index)) {
                            updatedChosenTemplate[1][1].content = block.attributes.content; // Heading
                        }

                        if ('myconianhotel/buttons' === block.name) {
                            updatedChosenTemplate[3][1].url = block.attributes.url;
                            updatedChosenTemplate[3][1].text = block.attributes.text;
                        }
                    });
                }


                replaceInnerBlocks(clientId, wp.blocks.createBlocksFromInnerBlocksTemplate(updatedChosenTemplate));
                lastTemplateRef.current = chosenTemplate;
            }
        }, [chosenTemplate, clientId]);

        // Set the initial background color if it hasn't been set
        const defaultBackgroundSlug  = useSetting('custom.defaultBackgroundSlug'); // Access custom theme setting
        useEffect(() => {
            if ( ! isNested) {
                if (! backgroundColor) {
                    setAttributes({ backgroundColor: defaultBackgroundSlug });
                }                
            } else {
                setAttributes({ backgroundColor: '' });
            }
        }, [backgroundColor]);

        const blockProps = useBlockProps({
            className: [
                'wp-block',
                ! isNested ? 'is-layout-constrained has-global-padding alignfull' : '',
                backgroundColor ? 'has-' + backgroundColor + '-background-color has-background-color' : '', // Color class
                chosenTemplate !== 'free' ? `is-style-${chosenTemplate}` : '',
            ].filter(Boolean).join(' ')
        });

        const innerBlocksProps = useInnerBlocksProps(
            { className: 'is-layout-flow' },
            {
                allowedBlocks,
                template: templates[chosenTemplate],
                templateLock: lockTemplate
            }
        );


        return (
            <>
                {displayPresets.length && (
                    <InspectorControls>
                        { ! isNested && (
                        <CustomPanelColorSettings
                            setAttributes={setAttributes}
                            backgroundColor={backgroundColor}
                            palette={'collection' == myconianhotelBlockFreeText.colorscheme ? ['lightgray', 'lightergray', 'white'] : ['darkblue', 'lightbrown', 'darkgray', 'lightgray', 'white'] }
                        />)}
                        <PanelBody title={__('Template Settings', 'myconianhotel')} initialOpen={true}>
                            <SelectControl
                                label={__('Select Template', 'myconianhotel')}
                                value={chosenTemplate}
                                options={options}
                                onChange={(newTemplate) => setAttributes({ chosenTemplate: newTemplate })}
                            />
                        </PanelBody>
                    </InspectorControls>
                )}
                <div {...blockProps}>
                    {/* Invisible editable field to focus on Editor's Blocks list */}
                    <textarea
                        style={{
                            border: '0',
                            clip: 'rect(1px, 1px, 1px, 1px)',
                            clipPath: 'inset(50%)',
                            height: '1px',
                            margin: '-1px',
                            overflow: 'hidden',
                            padding: '0',
                            position: 'absolute',
                            width: '1px',
                            wordWrap: 'normal'
                        }}
                    />
                    <div {...innerBlocksProps}></div>
                </div>
            </>
        );
    },
    save: (props) => {
        const { attributes } = props;
        const { backgroundColor } = attributes;

        const blockProps = useBlockProps.save({
            className: [
                'wp-block alignfull',
                ! props.attributes.isNested ? 'is-layout-constrained has-global-padding' : '',
                backgroundColor ? 'has-' + backgroundColor + '-background-color has-background-color' : '', // Color class
                props.attributes.chosenTemplate !== 'free' ? `is-style-${props.attributes.chosenTemplate}` : '',
            ].filter(Boolean).join(' ')
        });

        const innerBlocksProps = useInnerBlocksProps.save({ className: 'is-layout-flow' });

        return (
            <div {...blockProps}>
                <div {...innerBlocksProps}></div>
            </div>
        );
    },
});