// https://chatgpt.com/c/66f696cc-36bc-8003-947a-46033b527297

import { CustomPanelColorSettings, IconSet } from '../../modules';
import { onChangeMediaPosition } from '../../media-text/src/functions';
import MediaTextAttributes from '../../media-text/src/attributes';

const { __ } = wp.i18n;
const { registerBlockType, unregisterBlockStyle  } = wp.blocks;
const {
    InspectorControls,
    BlockControls, 
    useBlockProps,
    InnerBlocks,
    useInnerBlocksProps,
    useSetting 
} = wp.blockEditor;
const { TextControl } = wp.components; // UI components
const { 
    ToolbarButton, 
    ToolbarGroup
} = wp.components;
const { Fragment, useEffect, useState } = wp.element;

const { select, dispatch, useSelect } = wp.data;

const TEMPLATE = [
    ['myconianhotel/group', {
        className: 'wp-block-media-text__content is-layout-flow',
        'metadata': {
            'name': 'Content'
        }
    }, [
        ['myconianhotel/paragraph', {
            textAlign: 'center',
            fontSize: 'small',
            placeholder: __('Subtitle', 'myconianhotel')
        }],
        ['myconianhotel/heading', {
            level: 2,
            textAlign: 'center',
            placeholder: __('Heading', 'myconianhotel')
        }],
        ['myconianhotel/paragraph', {
            textAlign: 'center',
            placeholder: __('Content', 'myconianhotel')
        }],
        ['myconianhotel/buttons'],
        ['myconianhotel/paragraph', {
            textAlign: 'center',
            fontSize: 'xxlarge',
            className: 'myconianhotel-timeline',
            placeholder: __('Time', 'myconianhotel')
        }]
    ]],
    ['myconianhotel/gallery', { backgroundColor: 'transparent', sliderTheme: 'autowidth', aspectRatio: '20:37', layout: '', align: ''}]
];

const TEMPLATE_BOTTOM = [
    ['myconianhotel/group', {
        className: 'wp-block-media-text__content is-layout-flow',
        'metadata': {
            'name': 'Content'
        }
    }, [
        ['myconianhotel/paragraph', {
            textAlign: 'center',
            placeholder: __('Subtitle', 'myconianhotel')
        }],
        ['myconianhotel/heading', {
            level: 2,
            textAlign: 'center',
            placeholder: __('Heading', 'myconianhotel')
        }],
        ['myconianhotel/paragraph', {
            textAlign: 'center',
            placeholder: __('Content', 'myconianhotel')
        }]
    ]],
    ['myconianhotel/gallery', { backgroundColor: 'transparent', sliderTheme: 'autowidth', aspectRatio: 'unset', layout: 'constrained', align: ''}]
    ['myconianhotel/buttons']
];

const TEMPLATE_STICKY  = [
    ['myconianhotel/group', {
        className: 'wp-block-media-text__content is-layout-flow',
        'metadata': {
            'name': 'Content'
        }
    }, [
        ['myconianhotel/free-text', {displayPresets: ['free', 'presetMediaText'], chosenTemplate: 'presetMediaText', isNested: true}],
        ['myconianhotel/paragraph', {
            textAlign: 'center',
            fontSize: 'xxlarge',
            className: 'myconianhotel-timeline',
            placeholder: __('Time', 'myconianhotel')
        }]
    ]],
    ['myconianhotel/gallery', { backgroundColor: 'transparent', sliderTheme: 'autowidth', aspectRatio: '20:37', layout: '', align: ''}]
];

// Inline SVG icon as a string
const alignPullBottomIcon = (
    <svg class="myconianhotel-svg-icon" xmlns="http://www.w3.org/2000/svg" version="1.1" viewBox="0 0 20 20">
        <rect fill="#000000" x="5" y="3" width="10" height="2"/>
        <rect fill="#000000" x="3" y="11" width="14" height="6"/>
        <rect fill="#000000" x="5" y="7" width="10" height="2"/>
    </svg>
);    

registerBlockType('myconianhotel/gallery-text', {
    apiVersion: 2,
    title: __('Gallery & Text (M)'),
    description: __('A two-column layout with a gallery or video on one side and text on the other.'),
    icon: IconSet.galleryText,
    category: 'layout',
    attributes: {
        ...MediaTextAttributes,
        'timeline': {
            type: 'string',
            default: ''
        },
        backgroundColor: {
            type: 'string'
        },
    },
    supports: {
        //align: ['full'], // Only allow align full
        media: true,
        anchor: true/*,
        color: {
            background: true, // Native background color support
        }*/
    },
    edit: (props) => {
        const { attributes, setAttributes, clientId } = props;
        const { mediaPosition, backgroundColor, timeline } = attributes;
        const [template, setTemplate] = useState(mediaPosition === 'bottom' ? TEMPLATE_BOTTOM : (attributes.className?.includes('is-style-sticky') ? TEMPLATE_STICKY : TEMPLATE)); // State for controlling the link editing


        // Snipet to update existing blocks
        // Change presets to ['free', 'presetMediaText'] for existing Sticky Gallery text
        useEffect(() => {
            if (!attributes.className?.includes('is-style-sticky')) return;

            const innerBlocks = select('core/block-editor').getBlocks(clientId);
            const groupBlock = innerBlocks.find(block => block.name === 'myconianhotel/group');

            if (!groupBlock) return;

            const freeTextBlock = groupBlock.innerBlocks.find(block => block.name === 'myconianhotel/free-text');
            const timelineBlock = groupBlock.innerBlocks.find(block => block.attributes.className?.includes('myconianhotel-timeline'));

            if (
                freeTextBlock &&
                (!Array.isArray(freeTextBlock.attributes.displayPresets) ||
                !freeTextBlock.attributes.displayPresets.includes('presetMediaText'))
            ) {
                dispatch('core/block-editor').updateBlockAttributes(freeTextBlock.clientId, {
                    displayPresets: ['free', 'presetMediaText'],
                    isNested: true
                });
            }

            if (
                timelineBlock
            ) {
                dispatch('core/block-editor').updateBlockAttributes(timelineBlock.clientId, {
                    'metadata': {
                        'name': 'Time'
                    }
                });
            }
        }, [attributes.className, clientId]);

        // Dynamically unregister and register styles
        if (mediaPosition === 'bottom') {
            unregisterBlockStyle('myconianhotel/gallery-text', 'sticky');
        } else {
            wp.blocks.registerBlockStyle('myconianhotel/gallery-text', {
                name: 'sticky',
                label: __('Sticky Gallery', 'myconianhotel'),
            });
        }
        
        useEffect(() => {

            // Dynamically unregister and register styles
            if (mediaPosition === 'bottom') {
                unregisterBlockStyle('myconianhotel/gallery-text', 'sticky');
            } else {
                wp.blocks.registerBlockStyle('myconianhotel/gallery-text', {
                    name: 'sticky',
                    label: __('Sticky Gallery', 'myconianhotel'),
                });
            }
        }, [mediaPosition]);

        // Set the initial background color if it hasn't been set
        const defaultBackgroundSlug  = useSetting('custom.defaultBackgroundSlug'); // Access custom theme setting
        useEffect(() => {
            if (!backgroundColor) {
                setAttributes({ backgroundColor: defaultBackgroundSlug });
            }
        }, [backgroundColor]);

        // Monitor class name changes to adjust
        // sliderTheme attribute of nested gallery block and
        // align (with constrained class) attribute of this block
        useEffect(() => {
            const currentBlock = select('core/block-editor').getBlock(clientId);
            const isSticky = currentBlock?.attributes?.className?.includes('is-style-sticky');

            // Find nested `myconianhotel/gallery` block
            const innerBlocks = select('core/block-editor').getBlocks(clientId);
            const galleryBlock = innerBlocks.find(block => block.name === 'myconianhotel/gallery');

            if (galleryBlock) {
                // Update `sliderTheme` attribute based on sticky style
                dispatch('core/block-editor').updateBlockAttributes(galleryBlock.clientId, {
                    sliderTheme: isSticky ? '' : 'autowidth',
                    aspectRatio: 'unset'
                });
            }
        }, [attributes.className, clientId]);

        const blockProps = useBlockProps({
            className: [
                props.attributes.align ? `align${props.attributes.align}` : '', // Align classes
                attributes.className?.includes('is-style-sticky') || 'bottom' == mediaPosition ? 'has-global-padding' : '', // Media positioning class
                mediaPosition == 'bottom' || attributes.className?.includes('is-style-sticky') ? 'is-layout-constrained' : '', // Constrain class
                `has-${backgroundColor}-background-color has-background-color`,
                attributes.timeline ? 'has-timeline' : ''
            ].filter(Boolean).join(' ') // Filter out empty class values
        });

        const innerBlockProps = useInnerBlocksProps({
            className: [
                'wp-block-myconianhotel-media-text wp-block-media-text', // Default class
                'is-stacked-on-mobile', // Mobile stacking class                
                mediaPosition === 'right' ? 'has-media-on-the-right' : (mediaPosition === 'bottom' ? 'has-media-on-the-bottom' : ''), // Media positioning class
            ].filter(Boolean).join(' ') // Filter out empty class values
        }, {
            templateLock: 'all',
            template: template,
            allowedBlocks: []        
        });
        
        // Template depends on media position
        useEffect(() => {
            if ('bottom' == mediaPosition) {
                setTemplate(TEMPLATE_BOTTOM);
            } else {
                if (attributes.className?.includes('is-style-sticky')) {
                    setTemplate(TEMPLATE_STICKY);
                } else {
                    setTemplate(TEMPLATE);                    
                }
            }
        }, [mediaPosition]);
        
        useEffect(() => {
            if (attributes.className?.includes('is-style-sticky')) {
                setTemplate(TEMPLATE_STICKY);
            } else {
                setTemplate(TEMPLATE);                    
            }
        }, [attributes.className]);

        // Fetch inner blocks inside this parent block
        const innerBlocks = useSelect(
            (select) => select('core/block-editor').getBlocks(clientId),
            [clientId]
        );

        useEffect(() => {
            // Find `myconianhotel/group` inside the parent block
            const groupBlock = innerBlocks.find(block => block.name === 'myconianhotel/group');
            if (!groupBlock || !groupBlock.innerBlocks.length) return;

            // Get the last `myconianhotel/paragraph` inside `myconianhotel/group`
            const lastParagraphBlock = [...groupBlock.innerBlocks]
                .reverse()
                .find(block => block.name === 'myconianhotel/paragraph');

            // Update the parent's title attribute with the last paragraph content
            if (lastParagraphBlock) {
                setAttributes({ timeline: lastParagraphBlock.attributes.text });
            }
        }, [innerBlocks]);

        return (
            <Fragment>
                <BlockControls>
                    <ToolbarGroup>
                        <ToolbarButton
                            icon="align-pull-left"
                            label={__('Show media on left')}
                            isActive={mediaPosition === 'left'}
                            onClick={() => onChangeMediaPosition('left', setAttributes)}
                        />
                        <ToolbarButton
                            icon="align-pull-right"
                            label={__('Show media on right')}
                            isActive={mediaPosition === 'right'}
                            onClick={() => onChangeMediaPosition('right', setAttributes)}
                        />
                        <ToolbarButton
                            icon={alignPullBottomIcon}
                            label={__('Show media on bottom')}
                            isActive={mediaPosition === 'bottom'}
                            isDisabled={attributes.className?.includes('is-style-sticky')} // Disable if `is-style-sticky` class is present
                            onClick={() => onChangeMediaPosition('bottom', setAttributes)}
                        />                         
                    </ToolbarGroup>
                </BlockControls>
                <InspectorControls>
                    <CustomPanelColorSettings
                        setAttributes={setAttributes}
                        backgroundColor={backgroundColor}
                        palette={'collection' == myconianhotelBlockGalleryText.colorscheme ? ['lightgray', 'lightergray', 'white'] : ['darkblue', 'lightbrown', 'darkgray', 'lightgray', 'white'] }
                    />
                </InspectorControls>

                <div {...blockProps}>
                    {
                        false && attributes.className?.includes('is-style-sticky') &&
                        <div className="myconianhotel-timeline">
                            <TextControl
                                placeholder="Timeline"                                
                                value={timeline}
                                onChange={(value) => setAttributes({timeline: value})}
                            />
                        </div>
                    }
                    <div {...innerBlockProps}></div>
                </div>
            </Fragment>
        );
    },
    save: (props) => {
        const { attributes } = props;
        const { mediaPosition, backgroundColor } = attributes;

        const blockProps = useBlockProps.save({
            className: [
                props.attributes.align ? `align${props.attributes.align}` : '', // Align classes
                attributes.className?.includes('is-style-sticky') || 'bottom' == mediaPosition ? 'has-global-padding' : '', // Media positioning class
                mediaPosition == 'bottom' || attributes.className?.includes('is-style-sticky') ? 'is-layout-constrained' : '', // Constrain class
                `has-${backgroundColor}-background-color has-background-color`,
                attributes.timeline ? 'has-timeline' : ''
            ].filter(Boolean).join(' ') // Filter out empty class values
        });

        const innerBlockProps = useInnerBlocksProps.save({
            className: [
                'wp-block-myconianhotel-media-text wp-block-media-text', // Default class
                'is-stacked-on-mobile', // Mobile stacking class                
                mediaPosition === 'right' ? 'has-media-on-the-right' : (mediaPosition === 'bottom' ? 'has-media-on-the-bottom' : ''), // Media positioning class
            ].filter(Boolean).join(' ') // Filter out empty class values
        });

        return (
            <div {...blockProps}>
                {
                    false && attributes.timeline &&
                    <div className="myconianhotel-timeline has-xxlarge-font-size">{attributes.timeline}</div>
                }
                <div {...innerBlockProps}>
                    
                </div>
            </div>
        );
    }
});