const { __ } = wp.i18n;
const { registerBlockType } = wp.blocks; // Register blocks
const { useState, useEffect, useRef, Fragment } = wp.element; // React-like hooks in Gutenberg
const { TextControl, Button, PanelBody } = wp.components; // UI components
const { InspectorControls, useBlockProps, InnerBlocks, useInnerBlocksProps } = wp.blockEditor; // Block inspector controls
import { IconSet } from '../../modules';

registerBlockType('myconianhotel/google-map', {
  apiVersion: 2,
  title: 'Google Maps Block',
  icon: IconSet.googleMap,
  category: 'widgets',

  attributes: {
    markers: {
      type: 'array',
      default: [], // Default is an empty array
    }
  },

  edit: ({ attributes, setAttributes }) => {
    const { markers } = attributes;
    const [searchTerm, setSearchTerm] = useState(''); // State for search input
    const mapRef = useRef(null); // Reference to the map container
    const markersRef = useRef([]); // To store marker instances for map manipulation
    // SVG Icons
    const locationSvg = 
    `<svg xmlns="http://www.w3.org/2000/svg" width="40" height="40" viewBox="0 0 40 40">
        <g transform="translate(-1336 -3784)">
            <circle cx="20" cy="20" r="20" transform="translate(1336 3784)" fill="#013762"/>
            <path d="M8-22.381l2.62,6.069H5.4Zm5.132,11.844H15.4L8.919-25.416H7.91L1.4-10.538H2.929L4.9-15.132h6.227Z" transform="translate(1347.596 3821.977)" fill="#e9e7e4"/>
        </g>
    </svg>`;
    const pointsOfInterestSvg =
    `<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24">
      <circle cx="12" cy="12" r="12" fill="#b4aea1"/>
    </svg>`;

    const blockProps = useBlockProps({
        className: [
            'wp-block-media-text wp-block-myconianhotel-media-text wp-block-media-text has-media-on-the-right alignfull is-stacked-on-mobile', // Default class            
        ].filter(Boolean).join(' ') // Filter out empty class values
    });

    const TEMPLATE = [
        ['myconianhotel/paragraph', {
            fontSize: 'small',
            placeholder: __('Subtitle', 'myconianhotel')
        }],
        ['myconianhotel/heading', {
            level: 2,
            placeholder: __('Heading', 'myconianhotel')
        }]
    ];

    const innerBlocksProps = useInnerBlocksProps({}, {
        templateLock: 'all',
        template: TEMPLATE,
        allowedBlocks: []        
    });

    // Initialize Google Map when the block is mounted
    useEffect(() => {
      const googleMap = new google.maps.Map(mapRef.current, {
        center: { lat: 37.4414527, lng: 25.3463776 }, // Example coordinates
        mapId: "156839d38324d66d",
        zoom: 13.5,
        streetViewControl: false,
        fullscreenControl: false,
        mapTypeControl: false,
        zoomControl: false
      });
      mapRef.current.mapInstance = googleMap;

      // Add existing markers to the map
      markers.forEach((markerData, i) => {
        const iconSvg = 0 == i ? locationSvg : pointsOfInterestSvg
        const marker = new google.maps.Marker({
          position: markerData.position,
          map: googleMap,
          title: markerData.label || '', // Set the marker title to the label
          icon: `data:image/svg+xml;charset=UTF-8,${encodeURIComponent(iconSvg)}`
        });
        markersRef.current.push(marker);
      });
    }, []);

    // Function to add a marker
    const addMarker = () => {
      if (!searchTerm) return;

      const geocoder = new google.maps.Geocoder();
      geocoder.geocode({ address: searchTerm }, (results, status) => {
        if (status === 'OK') {
          const location = results[0].geometry.location;
          const newMarkerData = {
            position: { lat: location.lat(), lng: location.lng() },
            address: results[0].formatted_address,
            label: '', // Initialize label as empty
            icon: `data:image/svg+xml;charset=UTF-8,${encodeURIComponent(pointsOfInterestSvg)}`
          };

          const marker = new google.maps.Marker({
            position: location,
            map: mapRef.current.mapInstance,
            icon: `data:image/svg+xml;charset=UTF-8,${encodeURIComponent(pointsOfInterestSvg)}`
          });

          // Store the marker instance for manipulation (not serialized)
          markersRef.current.push(marker);

          // Update the markers attribute with serializable data
          setAttributes({ markers: [...markers, newMarkerData] });
          setSearchTerm(''); // Clear search input
        } else {
          alert('Location not found.');
        }
      });
    };

    // Function to update a marker's label
    const updateLabel = (index, newLabel) => {
      const updatedMarkers = markers.map((marker, i) =>
        i === index ? { ...marker, label: newLabel } : marker
      );
      console.log(updatedMarkers)
      setAttributes({ markers: updatedMarkers });
    };

    // Function to remove a marker
    const removeMarker = (index) => {
      const markerToRemove = markersRef.current[index];
      if (markerToRemove) {
        markerToRemove.setMap(null); // Remove marker from map
      }

      // Remove marker from the instance list
      markersRef.current = markersRef.current.filter((_, i) => i !== index);

      // Update markers attribute to remove the corresponding serialized data
      const updatedMarkers = markers.filter((_, i) => i !== index);
      setAttributes({ markers: updatedMarkers });
    };

    return (
      <Fragment>
        <InspectorControls>
            <PanelBody title="Markers Info">
                <p>Total Markers: {markers.length}</p>
            </PanelBody>
        </InspectorControls>

        <div {...blockProps}>
            <div className="wp-block-media-text__media" style={{ height: '80vh' }}>
                <div
                    ref={mapRef}
                    className="google-map-container"
                    style={{ height: '100%', width: '100%' }}
                    ></div>
            </div>        

            <div className="wp-block-media-text__content">
                {/*<InnerBlocks template={TEMPLATE} templateLock={'all'} />*/}
                <div {...innerBlocksProps}></div>

                <div>
                    <ul className="wp-block-myconianhotel-google-map__locations">
                        {markers.map((item, index) => (
                        <li key={index}>
                            <div>
                            <TextControl
                                placeholder="Label"
                                value={item.label}
                                onChange={(value) => updateLabel(index, value)}
                            />
                            <p className="has-small-font-size" style={{marginBottom: 0}}>{item.address}</p>
                            </div>
                            <Button
                                isDestructive
                                onClick={() => removeMarker(index)}
                                style={{ marginLeft: '10px' }}
                            >
                            Remove
                            </Button>
                        </li>
                        ))}
                        <li>
                            <TextControl
                                label="Search for a location"
                                placeholder="Enter a valid Google Map address"
                                value={searchTerm}
                                onChange={(value) => setSearchTerm(value)}
                            />
                            <Button isPrimary onClick={addMarker}>
                                Add Marker
                            </Button>                            
                        </li>
                    </ul>
                    
                </div>
            </div>
        </div>
    </Fragment>
    );
  },

  save: ({ attributes }) => {

    // Save the markers in a data attribute (optional)
    return (<InnerBlocks.Content />);
  },
});
