import MediaAttributes from './attributes';
import { MediaEdit, MediaSave, MediaUploadMedia, PanelBodyMedia } from './components';
import { IconSet } from '../../modules';

wp.domReady(() => {
    const { __ } = wp.i18n;
    const { registerBlockType } = wp.blocks;
    const { 
        InspectorControls,
        BlockControls, 
        useBlockProps,         
    } = wp.blockEditor;
    const { useSelect } = wp.data;    
    const { useEffect, Fragment } = wp.element;    

    registerBlockType('myconianhotel/media', {
        apiVersion: 2,
        title: __('Media (M)'),
        description: __('Insert an image or embed a video'),
        icon: IconSet.media,
        category: 'media',
        attributes: {
            ...MediaAttributes,
            align: {
                type: 'string',
                default: '',
            }
        },
        supports: {
            anchor: true,
            media: true,
            inserter: true, // Prevents this block from appearing in the block inserter
        },
        edit: (props) => {
            const { attributes, setAttributes } = props;
            const { 
                mediaId, 
                mediaUrl, 
                mediaAlt,
                mediaSizeSlug,
                mediaOrientation,
                mediaAnchor,
                relatedPostsLinkOption,
                relatedPosts,
                mediaCustomPosition,
                aspectRatio,
                originalAspectRatio,
                mediaType,
                autoplay,
                loop,
                muted,
                controls,
                playsinline,
                preload,
                poster,
                posterId,
                align
            } = attributes;

            const postId = useSelect((select) =>
                select('core/editor').getCurrentPostId(),
                []
            );

            // Snipet to update existing blocks 
            const media = useSelect( ( select ) =>
                select( 'core' ).getMedia( mediaId )
            );
            useEffect(() => {
                if ( ! originalAspectRatio && media?.media_details?.width && media?.media_details?.height ) {
                    setAttributes({originalAspectRatio: media.media_details.width / media.media_details.height})
                }
            }, []);

            // Initialize related post link
            // TODO: only for first initialization
            useEffect(() => {
                if (!mediaId || !relatedPostsLinkOption || !postId) return;

                const excludeIds = [postId]; // Exclude current
                fetch(`${window.wpApiSettings.root}myconianhotel/v1/image-usage/${mediaId}?exclude=${excludeIds.join(',')}`)
                    .then((res) => res.json())
                    .then((data) => {
                        setAttributes({ mediaAnchor: ! mediaAnchor && mediaAnchor !== '-1' ? data[0]?.value : mediaAnchor, relatedPosts: data });
                    });

            }, [mediaId]);

            // Construct block classNames
            const blockProps = useBlockProps({
                className: [
                    `wp-block-myconianhotel-image wp-block-${mediaType}`,
                    mediaOrientation ? 'has-' + mediaOrientation + '-image' : '',
                    align ? `align${align}` : ''
                ].filter(Boolean).join(' ') // Filter out empty class values
            });

            return (
                <Fragment>
                    <BlockControls>                        
                        <MediaUploadMedia
                            mediaId={mediaId}
                            setAttributes={setAttributes}
                        />
                    </BlockControls>
                    <InspectorControls>
                        <PanelBodyMedia
                            mediaType={mediaType}
                            mediaId={mediaId}
                            mediaAlt={mediaAlt}
                            mediaUrl={mediaUrl}
                            mediaSizeSlug={mediaSizeSlug}
                            mediaOrientation={mediaOrientation}
                            mediaAnchor={mediaAnchor}
                            relatedPostsLinkOption={relatedPostsLinkOption}
                            relatedPosts={relatedPosts}
                            mediaCustomPosition={mediaCustomPosition}
                            aspectRatio={aspectRatio}
                            originalAspectRatio={originalAspectRatio}
                            open={open}
                            controls={controls}
                            autoplay={autoplay}
                            loop={loop}
                            muted={muted}
                            playsinline={playsinline}
                            preload={preload}
                            poster={poster}
                            posterId={posterId}
                            setAttributes={setAttributes}
                        />
                    </InspectorControls>
                    <MediaEdit
                        mediaType={mediaType}
                        mediaUrl={mediaUrl}
                        mediaAlt={mediaAlt}
                        mediaId={mediaId}
                        mediaSizeSlug={mediaSizeSlug}
                        mediaOrientation={mediaOrientation}
                        mediaAnchor={mediaAnchor}
                        relatedPostsLinkOption={relatedPostsLinkOption}
                        relatedPosts={relatedPosts}
                        mediaCustomPosition={mediaCustomPosition}
                        aspectRatio={aspectRatio}
                        originalAspectRatio={originalAspectRatio}
                        controls={controls}
                        autoplay={autoplay}
                        loop={loop}
                        muted={muted}
                        playsinline={playsinline}
                        preload={preload}
                        poster={poster}
                        figureProps={blockProps}
                        setAttributes={setAttributes}
                    />
                </Fragment>
            );
        },
        save: (props) => {
            const { attributes } = props;
            const { 
                mediaId, 
                mediaUrl, 
                mediaAlt, 
                mediaSizeSlug,
                mediaOrientation,
                mediaAnchor,
                relatedPostsLinkOption,
                relatedPosts,
                mediaCustomPosition,
                aspectRatio,
                originalAspectRatio,
                mediaType,
                autoplay,
                loop,
                muted,
                controls,
                playsinline,
                preload,
                poster,
                align
            } = attributes;

            const blockProps = useBlockProps.save({
                className: [
                    `wp-block-myconianhotel-image wp-block-${attributes.mediaType}`,
                    mediaOrientation ? 'has-' + mediaOrientation + '-image' : '',
                    align ? `align${align}` : ''
                ].filter(Boolean).join(' ') // Filter out empty class values
            });

            return (
                <MediaSave
                    mediaType={mediaType}
                    mediaUrl={mediaUrl}
                    mediaAlt={mediaAlt}
                    mediaId={mediaId}
                    mediaSizeSlug={mediaSizeSlug}
                    mediaOrientation={mediaOrientation}
                    mediaAnchor={mediaAnchor}
                    relatedPostsLinkOption={relatedPostsLinkOption}
                    relatedPosts={relatedPosts}
                    mediaCustomPosition={mediaCustomPosition}
                    aspectRatio={aspectRatio}
                    originalAspectRatio={originalAspectRatio}
                    controls={controls}
                    autoplay={autoplay}
                    loop={loop}
                    muted={muted}
                    playsinline={playsinline}
                    preload={preload}
                    poster={poster}
                    figureProps={blockProps}
                />
            );
        }
    });
});